/*
SDL_ColorDef.h v0.9
by Jonathan Dearborn

Just a bunch of defines for SDL color calls.  This is so you can set colors with
quick, short calls.  The advantage to having them as defines is that they take
up absolutely no space in your program, regardless of number of colors, unless
you use them.

Typically, one would use SDL_MapRGB, like so:
Uint32 color = SDL_MapRGB(surface->format, red, green, blue);

With SDL_ColorDef, it looks like:
RGB_setSurface(surface);
Uint32 color = RGB_steelblue;

Or with alpha:
SDL_MapRGBA(surface->format, red, green, blue, alpha);
RGBA_lightgreen(alpha);

You get less typing and a specified color.

If you want to specify your own color, you can use:
Uint32 color = RGB_color(red, green, blue);
color = RGBA_color(red, green, blue, alpha);

Included are the W3C HTML colors.  They are all in lowercase to distinguish 
them.  Also, some fixes were made to these colors (found below the main block
of colors).
<Gray was darker than darkgray and lightgoldenrodyellow is unnecessarily long>

v0.9 2-17-10: Changed syntax and added RGB_Color(), RGB_setSurface(), and RGB_getSurface()
v0.8 8-15-08: Fixed my colors to look right.  Switched to RGB prefix.  Added 
  RGBA colors.  Added W3C web colors, found at...
http://www.w3schools.com/HTML/html_colornames.asp

Email me at grimfang4[at]gmail[dot]com if you'd like to add some
colors in the official copy.

*/

#ifndef _SDL_COLORDEF_H__
#define _SDL_COLORDEF_H__

#include "SDL.h"

#ifdef __cplusplus
extern "C"
{
#endif

void RGB_setSurface(SDL_Surface* surf);
#define RGB_SetSurface RGB_setSurface

void RGB_setFormat(SDL_PixelFormat* format);
SDL_PixelFormat* RGB_getFormat();

#define RGB_SetFormat RGB_setFormat
#define RGB_GetFormat RGB_getFormat

/* Any color */
#define RGB_Color(red, green, blue) SDL_MapRGB(RGB_getFormat(), red, green, blue)
#define RGBA_Color(red, green, blue, alpha) SDL_MapRGBA(RGB_getFormat(), red, green, blue, alpha)
#define RGB_color RGB_Color
#define RGBA_color RGBA_Color

/* W3C colors */
#define RGB_aliceblue SDL_MapRGB(RGB_getFormat(), 0xf0, 0xf8, 0xff)
#define RGB_antiquewhite SDL_MapRGB(RGB_getFormat(), 0xfa, 0xeb, 0xd7)
#define RGB_aqua SDL_MapRGB(RGB_getFormat(), 0x00, 0xff, 0xff)
#define RGB_aquamarine SDL_MapRGB(RGB_getFormat(), 0x7f, 0xff, 0xd4)
#define RGB_azure SDL_MapRGB(RGB_getFormat(), 0xf0, 0xff, 0xff)
#define RGB_beige SDL_MapRGB(RGB_getFormat(), 0xf5, 0xf5, 0xdc)
#define RGB_bisque SDL_MapRGB(RGB_getFormat(), 0xff, 0xe4, 0xc4)
#define RGB_black SDL_MapRGB(RGB_getFormat(), 0x00, 0x00, 0x00)
#define RGB_blanchedalmond SDL_MapRGB(RGB_getFormat(), 0xff, 0xeb, 0xcd)
#define RGB_blue SDL_MapRGB(RGB_getFormat(), 0x00, 0x00, 0xff)
#define RGB_blueviolet SDL_MapRGB(RGB_getFormat(), 0x8a, 0x2b, 0xe2)
#define RGB_brown SDL_MapRGB(RGB_getFormat(), 0xa5, 0x2a, 0x2a)
#define RGB_burlywood SDL_MapRGB(RGB_getFormat(), 0xde, 0xb8, 0x87)
#define RGB_cadetblue SDL_MapRGB(RGB_getFormat(), 0x5f, 0x9e, 0xa0)
#define RGB_chartreuse SDL_MapRGB(RGB_getFormat(), 0x7f, 0xff, 0x00)
#define RGB_chocolate SDL_MapRGB(RGB_getFormat(), 0xd2, 0x69, 0x1e)
#define RGB_coral SDL_MapRGB(RGB_getFormat(), 0xff, 0x7f, 0x50)
#define RGB_cornflowerblue SDL_MapRGB(RGB_getFormat(), 0x64, 0x95, 0xed)
#define RGB_cornsilk SDL_MapRGB(RGB_getFormat(), 0xff, 0xf8, 0xdc)
#define RGB_crimson SDL_MapRGB(RGB_getFormat(), 0xdc, 0x14, 0x3c)
#define RGB_cyan SDL_MapRGB(RGB_getFormat(), 0x00, 0xff, 0xff)
#define RGB_darkblue SDL_MapRGB(RGB_getFormat(), 0x00, 0x00, 0x8b)
#define RGB_darkcyan SDL_MapRGB(RGB_getFormat(), 0x00, 0x8b, 0x8b)
#define RGB_darkgoldenrod SDL_MapRGB(RGB_getFormat(), 0xb8, 0x86, 0x0b)
#define RGB_darkgreen SDL_MapRGB(RGB_getFormat(), 0x00, 0x64, 0x00)
#define RGB_darkkhaki SDL_MapRGB(RGB_getFormat(), 0xbd, 0xb7, 0x6b)
#define RGB_darkmagenta SDL_MapRGB(RGB_getFormat(), 0x8b, 0x00, 0x8b)
#define RGB_darkolivegreen SDL_MapRGB(RGB_getFormat(), 0x55, 0x6b, 0x2f)
#define RGB_darkorange SDL_MapRGB(RGB_getFormat(), 0xff, 0x8c, 0x00)
#define RGB_darkorchid SDL_MapRGB(RGB_getFormat(), 0x99, 0x32, 0xcc)
#define RGB_darkred SDL_MapRGB(RGB_getFormat(), 0x8b, 0x00, 0x00)
#define RGB_darksalmon SDL_MapRGB(RGB_getFormat(), 0xe9, 0x96, 0x7a)
#define RGB_darkseagreen SDL_MapRGB(RGB_getFormat(), 0x8f, 0xbc, 0x8f)
#define RGB_darkslateblue SDL_MapRGB(RGB_getFormat(), 0x48, 0x3d, 0x8b)
#define RGB_darkslategray SDL_MapRGB(RGB_getFormat(), 0x2f, 0x4f, 0x4f)
#define RGB_darkslategrey SDL_MapRGB(RGB_getFormat(), 0x2f, 0x4f, 0x4f)
#define RGB_darkturquoise SDL_MapRGB(RGB_getFormat(), 0x00, 0xce, 0xd1)
#define RGB_darkviolet SDL_MapRGB(RGB_getFormat(), 0x94, 0x00, 0xd3)
#define RGB_deeppink SDL_MapRGB(RGB_getFormat(), 0xff, 0x14, 0x93)
#define RGB_deepskyblue SDL_MapRGB(RGB_getFormat(), 0x00, 0xbf, 0xff)
#define RGB_dimgray SDL_MapRGB(RGB_getFormat(), 0x69, 0x69, 0x69)
#define RGB_dimgrey SDL_MapRGB(RGB_getFormat(), 0x69, 0x69, 0x69)
#define RGB_dodgerblue SDL_MapRGB(RGB_getFormat(), 0x1e, 0x90, 0xff)
#define RGB_firebrick SDL_MapRGB(RGB_getFormat(), 0xb2, 0x22, 0x22)
#define RGB_floralwhite SDL_MapRGB(RGB_getFormat(), 0xff, 0xfa, 0xf0)
#define RGB_forestgreen SDL_MapRGB(RGB_getFormat(), 0x22, 0x8b, 0x22)
#define RGB_fuchsia SDL_MapRGB(RGB_getFormat(), 0xff, 0x00, 0xff)
#define RGB_gainsboro SDL_MapRGB(RGB_getFormat(), 0xdc, 0xdc, 0xdc)
#define RGB_ghostwhite SDL_MapRGB(RGB_getFormat(), 0xf8, 0xf8, 0xff)
#define RGB_gold SDL_MapRGB(RGB_getFormat(), 0xff, 0xd7, 0x00)
#define RGB_goldenrod SDL_MapRGB(RGB_getFormat(), 0xda, 0xa5, 0x20)
#define RGB_green SDL_MapRGB(RGB_getFormat(), 0x00, 0x80, 0x00)
#define RGB_greenyellow SDL_MapRGB(RGB_getFormat(), 0xad, 0xff, 0x2f)
#define RGB_honeydew SDL_MapRGB(RGB_getFormat(), 0xf0, 0xff, 0xf0)
#define RGB_hotpink SDL_MapRGB(RGB_getFormat(), 0xff, 0x69, 0xb4)
#define RGB_indianred SDL_MapRGB(RGB_getFormat(), 0xcd, 0x5c, 0x5c)
#define RGB_indigo SDL_MapRGB(RGB_getFormat(), 0x4b, 0x00, 0x82)
#define RGB_ivory SDL_MapRGB(RGB_getFormat(), 0xff, 0xff, 0xf0)
#define RGB_khaki SDL_MapRGB(RGB_getFormat(), 0xf0, 0xe6, 0x8c)
#define RGB_lavender SDL_MapRGB(RGB_getFormat(), 0xe6, 0xe6, 0xfa)
#define RGB_lavenderblush SDL_MapRGB(RGB_getFormat(), 0xff, 0xf0, 0xf5)
#define RGB_lawngreen SDL_MapRGB(RGB_getFormat(), 0x7c, 0xfc, 0x00)
#define RGB_lemonchiffon SDL_MapRGB(RGB_getFormat(), 0xff, 0xfa, 0xcd)
#define RGB_lightblue SDL_MapRGB(RGB_getFormat(), 0xad, 0xd8, 0xe6)
#define RGB_lightcoral SDL_MapRGB(RGB_getFormat(), 0xf0, 0x80, 0x80)
#define RGB_lightcyan SDL_MapRGB(RGB_getFormat(), 0xe0, 0xff, 0xff)
#define RGB_lightgoldenrodyellow SDL_MapRGB(RGB_getFormat(), 0xfa, 0xfa, 0xd2)
#define RGB_lightgray SDL_MapRGB(RGB_getFormat(), 0xd3, 0xd3, 0xd3)
#define RGB_lightgrey SDL_MapRGB(RGB_getFormat(), 0xd3, 0xd3, 0xd3)
#define RGB_lightgreen SDL_MapRGB(RGB_getFormat(), 0x90, 0xee, 0x90)
#define RGB_lightpink SDL_MapRGB(RGB_getFormat(), 0xff, 0xb6, 0xc1)
#define RGB_lightsalmon SDL_MapRGB(RGB_getFormat(), 0xff, 0xa0, 0x7a)
#define RGB_lightseagreen SDL_MapRGB(RGB_getFormat(), 0x20, 0xb2, 0xaa)
#define RGB_lightskyblue SDL_MapRGB(RGB_getFormat(), 0x87, 0xce, 0xfa)
#define RGB_lightslategray SDL_MapRGB(RGB_getFormat(), 0x77, 0x88, 0x99)
#define RGB_lightslategrey SDL_MapRGB(RGB_getFormat(), 0x77, 0x88, 0x99)
#define RGB_lightsteelblue SDL_MapRGB(RGB_getFormat(), 0xb0, 0xc4, 0xde)
#define RGB_lightyellow SDL_MapRGB(RGB_getFormat(), 0xff, 0xff, 0xe0)
#define RGB_lime SDL_MapRGB(RGB_getFormat(), 0x00, 0xff, 0x00)
#define RGB_limegreen SDL_MapRGB(RGB_getFormat(), 0x32, 0xcd, 0x32)
#define RGB_linen SDL_MapRGB(RGB_getFormat(), 0xfa, 0xf0, 0xe6)
#define RGB_magenta SDL_MapRGB(RGB_getFormat(), 0xff, 0x00, 0xff)
#define RGB_maroon SDL_MapRGB(RGB_getFormat(), 0x80, 0x00, 0x00)
#define RGB_mediumaquamarine SDL_MapRGB(RGB_getFormat(), 0x66, 0xcd, 0xaa)
#define RGB_mediumblue SDL_MapRGB(RGB_getFormat(), 0x00, 0x00, 0xcd)
#define RGB_mediumorchid SDL_MapRGB(RGB_getFormat(), 0xba, 0x55, 0xd3)
#define RGB_mediumpurple SDL_MapRGB(RGB_getFormat(), 0x93, 0x70, 0xd8)
#define RGB_mediumseagreen SDL_MapRGB(RGB_getFormat(), 0x3c, 0xb3, 0x71)
#define RGB_mediumslateblue SDL_MapRGB(RGB_getFormat(), 0x7b, 0x68, 0xee)
#define RGB_mediumspringgreen SDL_MapRGB(RGB_getFormat(), 0x00, 0xfa, 0x9a)
#define RGB_mediumturquoise SDL_MapRGB(RGB_getFormat(), 0x48, 0xd1, 0xcc)
#define RGB_mediumvioletred SDL_MapRGB(RGB_getFormat(), 0xc7, 0x15, 0x85)
#define RGB_midnightblue SDL_MapRGB(RGB_getFormat(), 0x19, 0x19, 0x70)
#define RGB_mintcream SDL_MapRGB(RGB_getFormat(), 0xf5, 0xff, 0xfa)
#define RGB_mistyrose SDL_MapRGB(RGB_getFormat(), 0xff, 0xe4, 0xe1)
#define RGB_moccasin SDL_MapRGB(RGB_getFormat(), 0xff, 0xe4, 0xb5)
#define RGB_navajowhite SDL_MapRGB(RGB_getFormat(), 0xff, 0xde, 0xad)
#define RGB_navy SDL_MapRGB(RGB_getFormat(), 0x00, 0x00, 0x80)
#define RGB_oldlace SDL_MapRGB(RGB_getFormat(), 0xfd, 0xf5, 0xe6)
#define RGB_olive SDL_MapRGB(RGB_getFormat(), 0x80, 0x80, 0x00)
#define RGB_olivedrab SDL_MapRGB(RGB_getFormat(), 0x6b, 0x8e, 0x23)
#define RGB_orange SDL_MapRGB(RGB_getFormat(), 0xff, 0xa5, 0x00)
#define RGB_orangered SDL_MapRGB(RGB_getFormat(), 0xff, 0x45, 0x00)
#define RGB_orchid SDL_MapRGB(RGB_getFormat(), 0xda, 0x70, 0xd6)
#define RGB_palegoldenrod SDL_MapRGB(RGB_getFormat(), 0xee, 0xe8, 0xaa)
#define RGB_palegreen SDL_MapRGB(RGB_getFormat(), 0x98, 0xfb, 0x98)
#define RGB_paleturquoise SDL_MapRGB(RGB_getFormat(), 0xaf, 0xee, 0xee)
#define RGB_palevioletred SDL_MapRGB(RGB_getFormat(), 0xd8, 0x70, 0x93)
#define RGB_papayawhip SDL_MapRGB(RGB_getFormat(), 0xff, 0xef, 0xd5)
#define RGB_peachpuff SDL_MapRGB(RGB_getFormat(), 0xff, 0xda, 0xb9)
#define RGB_peru SDL_MapRGB(RGB_getFormat(), 0xcd, 0x85, 0x3f)
#define RGB_pink SDL_MapRGB(RGB_getFormat(), 0xff, 0xc0, 0xcb)
#define RGB_plum SDL_MapRGB(RGB_getFormat(), 0xdd, 0xa0, 0xdd)
#define RGB_powderblue SDL_MapRGB(RGB_getFormat(), 0xb0, 0xe0, 0xe6)
#define RGB_purple SDL_MapRGB(RGB_getFormat(), 0x80, 0x00, 0x80)
#define RGB_red SDL_MapRGB(RGB_getFormat(), 0xff, 0x00, 0x00)
#define RGB_rosybrown SDL_MapRGB(RGB_getFormat(), 0xbc, 0x8f, 0x8f)
#define RGB_royalblue SDL_MapRGB(RGB_getFormat(), 0x41, 0x69, 0xe1)
#define RGB_saddlebrown SDL_MapRGB(RGB_getFormat(), 0x8b, 0x45, 0x13)
#define RGB_salmon SDL_MapRGB(RGB_getFormat(), 0xfa, 0x80, 0x72)
#define RGB_sandybrown SDL_MapRGB(RGB_getFormat(), 0xf4, 0xa4, 0x60)
#define RGB_seagreen SDL_MapRGB(RGB_getFormat(), 0x2e, 0x8b, 0x57)
#define RGB_sienna SDL_MapRGB(RGB_getFormat(), 0xa0, 0x52, 0x2d)
#define RGB_silver SDL_MapRGB(RGB_getFormat(), 0xc0, 0xc0, 0xc0)
#define RGB_skyblue SDL_MapRGB(RGB_getFormat(), 0x87, 0xce, 0xeb)
#define RGB_slateblue SDL_MapRGB(RGB_getFormat(), 0x6a, 0x5a, 0xcd)
#define RGB_slategray SDL_MapRGB(RGB_getFormat(), 0x70, 0x80, 0x90)
#define RGB_slategrey SDL_MapRGB(RGB_getFormat(), 0x70, 0x80, 0x90)
#define RGB_snow SDL_MapRGB(RGB_getFormat(), 0xff, 0xfa, 0xfa)
#define RGB_springgreen SDL_MapRGB(RGB_getFormat(), 0x00, 0xff, 0x7f)
#define RGB_steelblue SDL_MapRGB(RGB_getFormat(), 0x46, 0x82, 0xb4)
#define RGB_tan SDL_MapRGB(RGB_getFormat(), 0xd2, 0xb4, 0x8c)
#define RGB_teal SDL_MapRGB(RGB_getFormat(), 0x00, 0x80, 0x80)
#define RGB_thistle SDL_MapRGB(RGB_getFormat(), 0xd8, 0xbf, 0xd8)
#define RGB_tomato SDL_MapRGB(RGB_getFormat(), 0xff, 0x63, 0x47)
#define RGB_turquoise SDL_MapRGB(RGB_getFormat(), 0x40, 0xe0, 0xd0)
#define RGB_violet SDL_MapRGB(RGB_getFormat(), 0xee, 0x82, 0xee)
#define RGB_wheat SDL_MapRGB(RGB_getFormat(), 0xf5, 0xde, 0xb3)
#define RGB_white SDL_MapRGB(RGB_getFormat(), 0xff, 0xff, 0xff)
#define RGB_whitesmoke SDL_MapRGB(RGB_getFormat(), 0xf5, 0xf5, 0xf5)
#define RGB_yellow SDL_MapRGB(RGB_getFormat(), 0xff, 0xff, 0x00)
#define RGB_yellowgreen SDL_MapRGB(RGB_getFormat(), 0x9a, 0xcd, 0x32)
/* Fixed W3C colors */
#define RGB_darkgray SDL_MapRGB(RGB_getFormat(), 0x80, 0x80, 0x80)
#define RGB_darkgrey SDL_MapRGB(RGB_getFormat(), 0x80, 0x80, 0x80)
#define RGB_gray SDL_MapRGB(RGB_getFormat(), 0xa9, 0xa9, 0xa9)
#define RGB_grey SDL_MapRGB(RGB_getFormat(), 0xa9, 0xa9, 0xa9)
#define RGB_lightgoldenrod SDL_MapRGB(RGB_getFormat(), 0xfa, 0xfa, 0xd2)



/* My colors */
#define RGB_REDVDK SDL_MapRGB(RGB_getFormat(), 64, 0, 0)
#define RGB_REDDK SDL_MapRGB(RGB_getFormat(), 128, 0, 0)
#define RGB_RED SDL_MapRGB(RGB_getFormat(), 255, 0, 0)
#define RGB_REDLT SDL_MapRGB(RGB_getFormat(), 255, 128, 128)
#define RGB_REDVLT SDL_MapRGB(RGB_getFormat(), 255, 192, 192)

#define RGB_GREENVDK SDL_MapRGB(RGB_getFormat(), 0, 64, 0)
#define RGB_GREENDK SDL_MapRGB(RGB_getFormat(), 0, 128, 0)
#define RGB_GREEN SDL_MapRGB(RGB_getFormat(), 0, 255, 0)
#define RGB_GREENLT SDL_MapRGB(RGB_getFormat(), 128, 255, 128)
#define RGB_GREENVLT SDL_MapRGB(RGB_getFormat(), 192, 255, 192)

#define RGB_BLUEVDK SDL_MapRGB(RGB_getFormat(), 0, 0, 64)
#define RGB_BLUEDK SDL_MapRGB(RGB_getFormat(), 0, 0, 128)
#define RGB_BLUE SDL_MapRGB(RGB_getFormat(), 0, 0, 255)
#define RGB_BLUELT SDL_MapRGB(RGB_getFormat(), 64, 128, 255)
#define RGB_BLUEVLT SDL_MapRGB(RGB_getFormat(), 96, 192, 255)

#define RGB_PERIWINKLEVDK SDL_MapRGB(RGB_getFormat(), 32, 32, 64)
#define RGB_PERIWINKLEDK SDL_MapRGB(RGB_getFormat(), 64, 64, 128)
#define RGB_PERIWINKLE SDL_MapRGB(RGB_getFormat(), 128, 128, 255)
#define RGB_PERIWINKLELT SDL_MapRGB(RGB_getFormat(), 192, 192, 255)
#define RGB_PERIWINKLEVLT SDL_MapRGB(RGB_getFormat(), 224, 224, 255)

#define RGB_LAVENDERVDK SDL_MapRGB(RGB_getFormat(), 96, 32, 160)
#define RGB_LAVENDERDK SDL_MapRGB(RGB_getFormat(), 128, 64, 192)
#define RGB_LAVENDER SDL_MapRGB(RGB_getFormat(), 160, 96, 212)
#define RGB_LAVENDERLT SDL_MapRGB(RGB_getFormat(), 192, 128, 255)
#define RGB_LAVENDERVLT SDL_MapRGB(RGB_getFormat(), 224, 160, 255)


#define RGB_PURPLEVDK SDL_MapRGB(RGB_getFormat(), 64, 0, 96)
#define RGB_PURPLEDK SDL_MapRGB(RGB_getFormat(), 96, 0, 128)
#define RGB_PURPLE SDL_MapRGB(RGB_getFormat(), 128, 0, 160)
#define RGB_PURPLELT SDL_MapRGB(RGB_getFormat(), 192, 64, 212)
#define RGB_PURPLEVLT SDL_MapRGB(RGB_getFormat(), 212, 128, 255)



#define RGB_BRIGHTPINK SDL_MapRGB(RGB_getFormat(), 255, 0, 255)

#define RGB_PINKVDK SDL_MapRGB(RGB_getFormat(), 128, 64, 96)
#define RGB_PINKDK SDL_MapRGB(RGB_getFormat(), 212, 92, 212)
#define RGB_PINK SDL_MapRGB(RGB_getFormat(), 255, 128, 255)
#define RGB_PINKLT SDL_MapRGB(RGB_getFormat(), 255, 160, 255)
#define RGB_PINKVLT SDL_MapRGB(RGB_getFormat(), 255, 192, 255)

#define RGB_YELLOWVDK SDL_MapRGB(RGB_getFormat(), 64, 64, 0)
#define RGB_YELLOWDK SDL_MapRGB(RGB_getFormat(), 128, 128, 0)
#define RGB_YELLOW SDL_MapRGB(RGB_getFormat(), 255, 255, 0)
#define RGB_YELLOWLT SDL_MapRGB(RGB_getFormat(), 255, 255, 128)
#define RGB_YELLOWVLT SDL_MapRGB(RGB_getFormat(), 255, 255, 192)

#define RGB_ORANGEVDK SDL_MapRGB(RGB_getFormat(), 180, 64, 0)
#define RGB_ORANGEDK SDL_MapRGB(RGB_getFormat(), 224, 96, 0)
#define RGB_ORANGE SDL_MapRGB(RGB_getFormat(), 255, 128, 0)
#define RGB_ORANGELT SDL_MapRGB(RGB_getFormat(), 255, 160, 32)
#define RGB_ORANGEVLT SDL_MapRGB(RGB_getFormat(), 255, 192, 64)

#define RGB_BROWNVDK SDL_MapRGB(RGB_getFormat(), 32, 16, 0)
#define RGB_BROWNDK SDL_MapRGB(RGB_getFormat(), 64, 32, 0)
#define RGB_BROWN SDL_MapRGB(RGB_getFormat(), 128, 64, 0)
#define RGB_BROWNLT SDL_MapRGB(RGB_getFormat(), 192, 128, 32)
#define RGB_BROWNVLT SDL_MapRGB(RGB_getFormat(), 255, 192, 64)

#define RGB_TEALVDK SDL_MapRGB(RGB_getFormat(), 0, 64, 64)
#define RGB_TEALDK SDL_MapRGB(RGB_getFormat(), 0, 128, 128)
#define RGB_TEAL SDL_MapRGB(RGB_getFormat(), 0, 255, 255)
#define RGB_TEALLT SDL_MapRGB(RGB_getFormat(), 128, 255, 255)
#define RGB_TEALVLT SDL_MapRGB(RGB_getFormat(), 192, 255, 255)

#define RGB_TURQUOISEVDK SDL_MapRGB(RGB_getFormat(), 0, 64, 64)
#define RGB_TURQUOISEDK SDL_MapRGB(RGB_getFormat(), 0, 128, 128)
#define RGB_TURQUOISE SDL_MapRGB(RGB_getFormat(), 0, 255, 255)
#define RGB_TURQUOISELT SDL_MapRGB(RGB_getFormat(), 128, 255, 255)
#define RGB_TURQUOISEVLT SDL_MapRGB(RGB_getFormat(), 192, 255, 255)

#define RGB_BLACK SDL_MapRGB(RGB_getFormat(), 0, 0, 0)
#define RGB_BLACK1 SDL_MapRGB(RGB_getFormat(), 1, 1, 1)
#define RGB_BLACK2 SDL_MapRGB(RGB_getFormat(), 2, 2, 2)

#define RGB_GRAYVDK SDL_MapRGB(RGB_getFormat(), 32, 32, 32)
#define RGB_GRAYDK SDL_MapRGB(RGB_getFormat(), 64, 64, 64)
#define RGB_GRAY SDL_MapRGB(RGB_getFormat(), 128, 128, 128)
#define RGB_GRAYLT SDL_MapRGB(RGB_getFormat(), 192, 192, 192)
#define RGB_GRAYVLT SDL_MapRGB(RGB_getFormat(), 224, 224, 224)

#define RGB_WHITE SDL_MapRGB(RGB_getFormat(), 255, 255, 255)
#define RGB_WHITE1 SDL_MapRGB(RGB_getFormat(), 254, 254, 254)
#define RGB_WHITE2 SDL_MapRGB(RGB_getFormat(), 253, 253, 253)
#define RGB_WHITEOFF SDL_MapRGB(RGB_getFormat(), 255, 255, 224)



// RGBA -----------------------------


/* W3C colors */
#define RGBA_aliceblue(alpha) SDL_MapRGBA(RGB_getFormat(), 0xf0, 0xf8, 0xff, alpha)
#define RGBA_antiquewhite(alpha) SDL_MapRGBA(RGB_getFormat(), 0xfa, 0xeb, 0xd7, alpha)
#define RGBA_aqua(alpha) SDL_MapRGBA(RGB_getFormat(), 0x00, 0xff, 0xff, alpha)
#define RGBA_aquamarine(alpha) SDL_MapRGBA(RGB_getFormat(), 0x7f, 0xff, 0xd4, alpha)
#define RGBA_azure(alpha) SDL_MapRGBA(RGB_getFormat(), 0xf0, 0xff, 0xff, alpha)
#define RGBA_beige(alpha) SDL_MapRGBA(RGB_getFormat(), 0xf5, 0xf5, 0xdc, alpha)
#define RGBA_bisque(alpha) SDL_MapRGBA(RGB_getFormat(), 0xff, 0xe4, 0xc4, alpha)
#define RGBA_black(alpha) SDL_MapRGBA(RGB_getFormat(), 0x00, 0x00, 0x00, alpha)
#define RGBA_blanchedalmond(alpha) SDL_MapRGBA(RGB_getFormat(), 0xff, 0xeb, 0xcd, alpha)
#define RGBA_blue(alpha) SDL_MapRGBA(RGB_getFormat(), 0x00, 0x00, 0xff, alpha)
#define RGBA_blueviolet(alpha) SDL_MapRGBA(RGB_getFormat(), 0x8a, 0x2b, 0xe2, alpha)
#define RGBA_brown(alpha) SDL_MapRGBA(RGB_getFormat(), 0xa5, 0x2a, 0x2a, alpha)
#define RGBA_burlywood(alpha) SDL_MapRGBA(RGB_getFormat(), 0xde, 0xb8, 0x87, alpha)
#define RGBA_cadetblue(alpha) SDL_MapRGBA(RGB_getFormat(), 0x5f, 0x9e, 0xa0, alpha)
#define RGBA_chartreuse(alpha) SDL_MapRGBA(RGB_getFormat(), 0x7f, 0xff, 0x00, alpha)
#define RGBA_chocolate(alpha) SDL_MapRGBA(RGB_getFormat(), 0xd2, 0x69, 0x1e, alpha)
#define RGBA_coral(alpha) SDL_MapRGBA(RGB_getFormat(), 0xff, 0x7f, 0x50, alpha)
#define RGBA_cornflowerblue(alpha) SDL_MapRGBA(RGB_getFormat(), 0x64, 0x95, 0xed, alpha)
#define RGBA_cornsilk(alpha) SDL_MapRGBA(RGB_getFormat(), 0xff, 0xf8, 0xdc, alpha)
#define RGBA_crimson(alpha) SDL_MapRGBA(RGB_getFormat(), 0xdc, 0x14, 0x3c, alpha)
#define RGBA_cyan(alpha) SDL_MapRGBA(RGB_getFormat(), 0x00, 0xff, 0xff, alpha)
#define RGBA_darkblue(alpha) SDL_MapRGBA(RGB_getFormat(), 0x00, 0x00, 0x8b, alpha)
#define RGBA_darkcyan(alpha) SDL_MapRGBA(RGB_getFormat(), 0x00, 0x8b, 0x8b, alpha)
#define RGBA_darkgoldenrod(alpha) SDL_MapRGBA(RGB_getFormat(), 0xb8, 0x86, 0x0b, alpha)
#define RGBA_darkgreen(alpha) SDL_MapRGBA(RGB_getFormat(), 0x00, 0x64, 0x00, alpha)
#define RGBA_darkkhaki(alpha) SDL_MapRGBA(RGB_getFormat(), 0xbd, 0xb7, 0x6b, alpha)
#define RGBA_darkmagenta(alpha) SDL_MapRGBA(RGB_getFormat(), 0x8b, 0x00, 0x8b, alpha)
#define RGBA_darkolivegreen(alpha) SDL_MapRGBA(RGB_getFormat(), 0x55, 0x6b, 0x2f, alpha)
#define RGBA_darkorange(alpha) SDL_MapRGBA(RGB_getFormat(), 0xff, 0x8c, 0x00, alpha)
#define RGBA_darkorchid(alpha) SDL_MapRGBA(RGB_getFormat(), 0x99, 0x32, 0xcc, alpha)
#define RGBA_darkred(alpha) SDL_MapRGBA(RGB_getFormat(), 0x8b, 0x00, 0x00, alpha)
#define RGBA_darksalmon(alpha) SDL_MapRGBA(RGB_getFormat(), 0xe9, 0x96, 0x7a, alpha)
#define RGBA_darkseagreen(alpha) SDL_MapRGBA(RGB_getFormat(), 0x8f, 0xbc, 0x8f, alpha)
#define RGBA_darkslateblue(alpha) SDL_MapRGBA(RGB_getFormat(), 0x48, 0x3d, 0x8b, alpha)
#define RGBA_darkslategray(alpha) SDL_MapRGBA(RGB_getFormat(), 0x2f, 0x4f, 0x4f, alpha)
#define RGBA_darkslategrey(alpha) SDL_MapRGBA(RGB_getFormat(), 0x2f, 0x4f, 0x4f, alpha)
#define RGBA_darkturquoise(alpha) SDL_MapRGBA(RGB_getFormat(), 0x00, 0xce, 0xd1, alpha)
#define RGBA_darkviolet(alpha) SDL_MapRGBA(RGB_getFormat(), 0x94, 0x00, 0xd3, alpha)
#define RGBA_deeppink(alpha) SDL_MapRGBA(RGB_getFormat(), 0xff, 0x14, 0x93, alpha)
#define RGBA_deepskyblue(alpha) SDL_MapRGBA(RGB_getFormat(), 0x00, 0xbf, 0xff, alpha)
#define RGBA_dimgray(alpha) SDL_MapRGBA(RGB_getFormat(), 0x69, 0x69, 0x69, alpha)
#define RGBA_dimgrey(alpha) SDL_MapRGBA(RGB_getFormat(), 0x69, 0x69, 0x69, alpha)
#define RGBA_dodgerblue(alpha) SDL_MapRGBA(RGB_getFormat(), 0x1e, 0x90, 0xff, alpha)
#define RGBA_firebrick(alpha) SDL_MapRGBA(RGB_getFormat(), 0xb2, 0x22, 0x22, alpha)
#define RGBA_floralwhite(alpha) SDL_MapRGBA(RGB_getFormat(), 0xff, 0xfa, 0xf0, alpha)
#define RGBA_forestgreen(alpha) SDL_MapRGBA(RGB_getFormat(), 0x22, 0x8b, 0x22, alpha)
#define RGBA_fuchsia(alpha) SDL_MapRGBA(RGB_getFormat(), 0xff, 0x00, 0xff, alpha)
#define RGBA_gainsboro(alpha) SDL_MapRGBA(RGB_getFormat(), 0xdc, 0xdc, 0xdc, alpha)
#define RGBA_ghostwhite(alpha) SDL_MapRGBA(RGB_getFormat(), 0xf8, 0xf8, 0xff, alpha)
#define RGBA_gold(alpha) SDL_MapRGBA(RGB_getFormat(), 0xff, 0xd7, 0x00, alpha)
#define RGBA_goldenrod(alpha) SDL_MapRGBA(RGB_getFormat(), 0xda, 0xa5, 0x20, alpha)
#define RGBA_green(alpha) SDL_MapRGBA(RGB_getFormat(), 0x00, 0x80, 0x00, alpha)
#define RGBA_greenyellow(alpha) SDL_MapRGBA(RGB_getFormat(), 0xad, 0xff, 0x2f, alpha)
#define RGBA_honeydew(alpha) SDL_MapRGBA(RGB_getFormat(), 0xf0, 0xff, 0xf0, alpha)
#define RGBA_hotpink(alpha) SDL_MapRGBA(RGB_getFormat(), 0xff, 0x69, 0xb4, alpha)
#define RGBA_indianred(alpha) SDL_MapRGBA(RGB_getFormat(), 0xcd, 0x5c, 0x5c, alpha)
#define RGBA_indigo(alpha) SDL_MapRGBA(RGB_getFormat(), 0x4b, 0x00, 0x82, alpha)
#define RGBA_ivory(alpha) SDL_MapRGBA(RGB_getFormat(), 0xff, 0xff, 0xf0, alpha)
#define RGBA_khaki(alpha) SDL_MapRGBA(RGB_getFormat(), 0xf0, 0xe6, 0x8c, alpha)
#define RGBA_lavender(alpha) SDL_MapRGBA(RGB_getFormat(), 0xe6, 0xe6, 0xfa, alpha)
#define RGBA_lavenderblush(alpha) SDL_MapRGBA(RGB_getFormat(), 0xff, 0xf0, 0xf5, alpha)
#define RGBA_lawngreen(alpha) SDL_MapRGBA(RGB_getFormat(), 0x7c, 0xfc, 0x00, alpha)
#define RGBA_lemonchiffon(alpha) SDL_MapRGBA(RGB_getFormat(), 0xff, 0xfa, 0xcd, alpha)
#define RGBA_lightblue(alpha) SDL_MapRGBA(RGB_getFormat(), 0xad, 0xd8, 0xe6, alpha)
#define RGBA_lightcoral(alpha) SDL_MapRGBA(RGB_getFormat(), 0xf0, 0x80, 0x80, alpha)
#define RGBA_lightcyan(alpha) SDL_MapRGBA(RGB_getFormat(), 0xe0, 0xff, 0xff, alpha)
#define RGBA_lightgoldenrodyellow(alpha) SDL_MapRGBA(RGB_getFormat(), 0xfa, 0xfa, 0xd2, alpha)
#define RGBA_lightgray(alpha) SDL_MapRGBA(RGB_getFormat(), 0xd3, 0xd3, 0xd3, alpha)
#define RGBA_lightgrey(alpha) SDL_MapRGBA(RGB_getFormat(), 0xd3, 0xd3, 0xd3, alpha)
#define RGBA_lightgreen(alpha) SDL_MapRGBA(RGB_getFormat(), 0x90, 0xee, 0x90, alpha)
#define RGBA_lightpink(alpha) SDL_MapRGBA(RGB_getFormat(), 0xff, 0xb6, 0xc1, alpha)
#define RGBA_lightsalmon(alpha) SDL_MapRGBA(RGB_getFormat(), 0xff, 0xa0, 0x7a, alpha)
#define RGBA_lightseagreen(alpha) SDL_MapRGBA(RGB_getFormat(), 0x20, 0xb2, 0xaa, alpha)
#define RGBA_lightskyblue(alpha) SDL_MapRGBA(RGB_getFormat(), 0x87, 0xce, 0xfa, alpha)
#define RGBA_lightslategray(alpha) SDL_MapRGBA(RGB_getFormat(), 0x77, 0x88, 0x99, alpha)
#define RGBA_lightslategrey(alpha) SDL_MapRGBA(RGB_getFormat(), 0x77, 0x88, 0x99, alpha)
#define RGBA_lightsteelblue(alpha) SDL_MapRGBA(RGB_getFormat(), 0xb0, 0xc4, 0xde, alpha)
#define RGBA_lightyellow(alpha) SDL_MapRGBA(RGB_getFormat(), 0xff, 0xff, 0xe0, alpha)
#define RGBA_lime(alpha) SDL_MapRGBA(RGB_getFormat(), 0x00, 0xff, 0x00, alpha)
#define RGBA_limegreen(alpha) SDL_MapRGBA(RGB_getFormat(), 0x32, 0xcd, 0x32, alpha)
#define RGBA_linen(alpha) SDL_MapRGBA(RGB_getFormat(), 0xfa, 0xf0, 0xe6, alpha)
#define RGBA_magenta(alpha) SDL_MapRGBA(RGB_getFormat(), 0xff, 0x00, 0xff, alpha)
#define RGBA_maroon(alpha) SDL_MapRGBA(RGB_getFormat(), 0x80, 0x00, 0x00, alpha)
#define RGBA_mediumaquamarine(alpha) SDL_MapRGBA(RGB_getFormat(), 0x66, 0xcd, 0xaa, alpha)
#define RGBA_mediumblue(alpha) SDL_MapRGBA(RGB_getFormat(), 0x00, 0x00, 0xcd, alpha)
#define RGBA_mediumorchid(alpha) SDL_MapRGBA(RGB_getFormat(), 0xba, 0x55, 0xd3, alpha)
#define RGBA_mediumpurple(alpha) SDL_MapRGBA(RGB_getFormat(), 0x93, 0x70, 0xd8, alpha)
#define RGBA_mediumseagreen(alpha) SDL_MapRGBA(RGB_getFormat(), 0x3c, 0xb3, 0x71, alpha)
#define RGBA_mediumslateblue(alpha) SDL_MapRGBA(RGB_getFormat(), 0x7b, 0x68, 0xee, alpha)
#define RGBA_mediumspringgreen(alpha) SDL_MapRGBA(RGB_getFormat(), 0x00, 0xfa, 0x9a, alpha)
#define RGBA_mediumturquoise(alpha) SDL_MapRGBA(RGB_getFormat(), 0x48, 0xd1, 0xcc, alpha)
#define RGBA_mediumvioletred(alpha) SDL_MapRGBA(RGB_getFormat(), 0xc7, 0x15, 0x85, alpha)
#define RGBA_midnightblue(alpha) SDL_MapRGBA(RGB_getFormat(), 0x19, 0x19, 0x70, alpha)
#define RGBA_mintcream(alpha) SDL_MapRGBA(RGB_getFormat(), 0xf5, 0xff, 0xfa, alpha)
#define RGBA_mistyrose(alpha) SDL_MapRGBA(RGB_getFormat(), 0xff, 0xe4, 0xe1, alpha)
#define RGBA_moccasin(alpha) SDL_MapRGBA(RGB_getFormat(), 0xff, 0xe4, 0xb5, alpha)
#define RGBA_navajowhite(alpha) SDL_MapRGBA(RGB_getFormat(), 0xff, 0xde, 0xad, alpha)
#define RGBA_navy(alpha) SDL_MapRGBA(RGB_getFormat(), 0x00, 0x00, 0x80, alpha)
#define RGBA_oldlace(alpha) SDL_MapRGBA(RGB_getFormat(), 0xfd, 0xf5, 0xe6, alpha)
#define RGBA_olive(alpha) SDL_MapRGBA(RGB_getFormat(), 0x80, 0x80, 0x00, alpha)
#define RGBA_olivedrab(alpha) SDL_MapRGBA(RGB_getFormat(), 0x6b, 0x8e, 0x23, alpha)
#define RGBA_orange(alpha) SDL_MapRGBA(RGB_getFormat(), 0xff, 0xa5, 0x00, alpha)
#define RGBA_orangered(alpha) SDL_MapRGBA(RGB_getFormat(), 0xff, 0x45, 0x00, alpha)
#define RGBA_orchid(alpha) SDL_MapRGBA(RGB_getFormat(), 0xda, 0x70, 0xd6, alpha)
#define RGBA_palegoldenrod(alpha) SDL_MapRGBA(RGB_getFormat(), 0xee, 0xe8, 0xaa, alpha)
#define RGBA_palegreen(alpha) SDL_MapRGBA(RGB_getFormat(), 0x98, 0xfb, 0x98, alpha)
#define RGBA_paleturquoise(alpha) SDL_MapRGBA(RGB_getFormat(), 0xaf, 0xee, 0xee, alpha)
#define RGBA_palevioletred(alpha) SDL_MapRGBA(RGB_getFormat(), 0xd8, 0x70, 0x93, alpha)
#define RGBA_papayawhip(alpha) SDL_MapRGBA(RGB_getFormat(), 0xff, 0xef, 0xd5, alpha)
#define RGBA_peachpuff(alpha) SDL_MapRGBA(RGB_getFormat(), 0xff, 0xda, 0xb9, alpha)
#define RGBA_peru(alpha) SDL_MapRGBA(RGB_getFormat(), 0xcd, 0x85, 0x3f, alpha)
#define RGBA_pink(alpha) SDL_MapRGBA(RGB_getFormat(), 0xff, 0xc0, 0xcb, alpha)
#define RGBA_plum(alpha) SDL_MapRGBA(RGB_getFormat(), 0xdd, 0xa0, 0xdd, alpha)
#define RGBA_powderblue(alpha) SDL_MapRGBA(RGB_getFormat(), 0xb0, 0xe0, 0xe6, alpha)
#define RGBA_purple(alpha) SDL_MapRGBA(RGB_getFormat(), 0x80, 0x00, 0x80, alpha)
#define RGBA_red(alpha) SDL_MapRGBA(RGB_getFormat(), 0xff, 0x00, 0x00, alpha)
#define RGBA_rosybrown(alpha) SDL_MapRGBA(RGB_getFormat(), 0xbc, 0x8f, 0x8f, alpha)
#define RGBA_royalblue(alpha) SDL_MapRGBA(RGB_getFormat(), 0x41, 0x69, 0xe1, alpha)
#define RGBA_saddlebrown(alpha) SDL_MapRGBA(RGB_getFormat(), 0x8b, 0x45, 0x13, alpha)
#define RGBA_salmon(alpha) SDL_MapRGBA(RGB_getFormat(), 0xfa, 0x80, 0x72, alpha)
#define RGBA_sandybrown(alpha) SDL_MapRGBA(RGB_getFormat(), 0xf4, 0xa4, 0x60, alpha)
#define RGBA_seagreen(alpha) SDL_MapRGBA(RGB_getFormat(), 0x2e, 0x8b, 0x57, alpha)
#define RGBA_sienna(alpha) SDL_MapRGBA(RGB_getFormat(), 0xa0, 0x52, 0x2d, alpha)
#define RGBA_silver(alpha) SDL_MapRGBA(RGB_getFormat(), 0xc0, 0xc0, 0xc0, alpha)
#define RGBA_skyblue(alpha) SDL_MapRGBA(RGB_getFormat(), 0x87, 0xce, 0xeb, alpha)
#define RGBA_slateblue(alpha) SDL_MapRGBA(RGB_getFormat(), 0x6a, 0x5a, 0xcd, alpha)
#define RGBA_slategray(alpha) SDL_MapRGBA(RGB_getFormat(), 0x70, 0x80, 0x90, alpha)
#define RGBA_slategrey(alpha) SDL_MapRGBA(RGB_getFormat(), 0x70, 0x80, 0x90, alpha)
#define RGBA_snow(alpha) SDL_MapRGBA(RGB_getFormat(), 0xff, 0xfa, 0xfa, alpha)
#define RGBA_springgreen(alpha) SDL_MapRGBA(RGB_getFormat(), 0x00, 0xff, 0x7f, alpha)
#define RGBA_steelblue(alpha) SDL_MapRGBA(RGB_getFormat(), 0x46, 0x82, 0xb4, alpha)
#define RGBA_tan(alpha) SDL_MapRGBA(RGB_getFormat(), 0xd2, 0xb4, 0x8c, alpha)
#define RGBA_teal(alpha) SDL_MapRGBA(RGB_getFormat(), 0x00, 0x80, 0x80, alpha)
#define RGBA_thistle(alpha) SDL_MapRGBA(RGB_getFormat(), 0xd8, 0xbf, 0xd8, alpha)
#define RGBA_tomato(alpha) SDL_MapRGBA(RGB_getFormat(), 0xff, 0x63, 0x47, alpha)
#define RGBA_turquoise(alpha) SDL_MapRGBA(RGB_getFormat(), 0x40, 0xe0, 0xd0, alpha)
#define RGBA_violet(alpha) SDL_MapRGBA(RGB_getFormat(), 0xee, 0x82, 0xee, alpha)
#define RGBA_wheat(alpha) SDL_MapRGBA(RGB_getFormat(), 0xf5, 0xde, 0xb3, alpha)
#define RGBA_white(alpha) SDL_MapRGBA(RGB_getFormat(), 0xff, 0xff, 0xff, alpha)
#define RGBA_whitesmoke(alpha) SDL_MapRGBA(RGB_getFormat(), 0xf5, 0xf5, 0xf5, alpha)
#define RGBA_yellow(alpha) SDL_MapRGBA(RGB_getFormat(), 0xff, 0xff, 0x00, alpha)
#define RGBA_yellowgreen(alpha) SDL_MapRGBA(RGB_getFormat(), 0x9a, 0xcd, 0x32, alpha)
/* Fixed W3C colors */
#define RGBA_darkgray(alpha) SDL_MapRGBA(RGB_getFormat(), 0x80, 0x80, 0x80, alpha)
#define RGBA_darkgrey(alpha) SDL_MapRGBA(RGB_getFormat(), 0x80, 0x80, 0x80, alpha)
#define RGBA_gray(alpha) SDL_MapRGBA(RGB_getFormat(), 0xa9, 0xa9, 0xa9, alpha)
#define RGBA_grey(alpha) SDL_MapRGBA(RGB_getFormat(), 0xa9, 0xa9, 0xa9, alpha)
#define RGBA_lightgoldenrod(alpha) SDL_MapRGBA(RGB_getFormat(), 0xfa, 0xfa, 0xd2, alpha)





/* My Colors*/


#define RGBA_REDVDK(alpha) SDL_MapRGBA(RGB_getFormat(), 64, 0, 0, alpha)
#define RGBA_REDDK(alpha) SDL_MapRGBA(RGB_getFormat(), 128, 0, 0, alpha)
#define RGBA_RED(alpha) SDL_MapRGBA(RGB_getFormat(), 255, 0, 0, alpha)
#define RGBA_REDLT(alpha) SDL_MapRGBA(RGB_getFormat(), 255, 128, 128, alpha)
#define RGBA_REDVLT(alpha) SDL_MapRGBA(RGB_getFormat(), 255, 192, 192, alpha)

#define RGBA_GREENVDK(alpha) SDL_MapRGBA(RGB_getFormat(), 0, 64, 0, alpha)
#define RGBA_GREENDK(alpha) SDL_MapRGBA(RGB_getFormat(), 0, 128, 0, alpha)
#define RGBA_GREEN(alpha) SDL_MapRGBA(RGB_getFormat(), 0, 255, 0, alpha)
#define RGBA_GREENLT(alpha) SDL_MapRGBA(RGB_getFormat(), 128, 255, 128, alpha)
#define RGBA_GREENVLT(alpha) SDL_MapRGBA(RGB_getFormat(), 192, 255, 192, alpha)

#define RGBA_BLUEVDK(alpha) SDL_MapRGBA(RGB_getFormat(), 0, 0, 64, alpha)
#define RGBA_BLUEDK(alpha) SDL_MapRGBA(RGB_getFormat(), 0, 0, 128, alpha)
#define RGBA_BLUE(alpha) SDL_MapRGBA(RGB_getFormat(), 0, 0, 255, alpha)
#define RGBA_BLUELT(alpha) SDL_MapRGBA(RGB_getFormat(), 64, 128, 255, alpha)
#define RGBA_BLUEVLT(alpha) SDL_MapRGBA(RGB_getFormat(), 96, 192, 255, alpha)

#define RGBA_PERIWINKLEVDK(alpha) SDL_MapRGBA(RGB_getFormat(), 32, 32, 64, alpha)
#define RGBA_PERIWINKLEDK(alpha) SDL_MapRGBA(RGB_getFormat(), 64, 64, 128, alpha)
#define RGBA_PERIWINKLE(alpha) SDL_MapRGBA(RGB_getFormat(), 128, 128, 255, alpha)
#define RGBA_PERIWINKLELT(alpha) SDL_MapRGBA(RGB_getFormat(), 192, 192, 255, alpha)
#define RGBA_PERIWINKLEVLT(alpha) SDL_MapRGBA(RGB_getFormat(), 224, 224, 255, alpha)

#define RGBA_LAVENDERVDK(alpha) SDL_MapRGBA(RGB_getFormat(), 96, 32, 160, alpha)
#define RGBA_LAVENDERDK(alpha) SDL_MapRGBA(RGB_getFormat(), 128, 64, 192, alpha)
#define RGBA_LAVENDER(alpha) SDL_MapRGBA(RGB_getFormat(), 160, 96, 212, alpha)
#define RGBA_LAVENDERLT(alpha) SDL_MapRGBA(RGB_getFormat(), 192, 128, 255, alpha)
#define RGBA_LAVENDERVLT(alpha) SDL_MapRGBA(RGB_getFormat(), 224, 160, 255, alpha)


#define RGBA_PURPLEVDK(alpha) SDL_MapRGBA(RGB_getFormat(), 64, 0, 96, alpha)
#define RGBA_PURPLEDK(alpha) SDL_MapRGBA(RGB_getFormat(), 96, 0, 128, alpha)
#define RGBA_PURPLE(alpha) SDL_MapRGBA(RGB_getFormat(), 128, 0, 160, alpha)
#define RGBA_PURPLELT(alpha) SDL_MapRGBA(RGB_getFormat(), 192, 64, 212, alpha)
#define RGBA_PURPLEVLT(alpha) SDL_MapRGBA(RGB_getFormat(), 212, 128, 255, alpha)

#define RGBA_BRIGHTPINK(alpha) SDL_MapRGBA(RGB_getFormat(), 255, 0, 255, alpha)

#define RGBA_PINKVDK(alpha) SDL_MapRGBA(RGB_getFormat(), 64, 32, 64, alpha)
#define RGBA_PINKDK(alpha) SDL_MapRGBA(RGB_getFormat(), 128, 64, 128, alpha)
#define RGBA_PINK(alpha) SDL_MapRGBA(RGB_getFormat(), 255, 128, 255, alpha)
#define RGBA_PINKLT(alpha) SDL_MapRGBA(RGB_getFormat(), 255, 192, 255, alpha)
#define RGBA_PINKVLT(alpha) SDL_MapRGBA(RGB_getFormat(), 255, 224, 255, alpha)

#define RGBA_YELLOWVDK(alpha) SDL_MapRGBA(RGB_getFormat(), 64, 64, 0, alpha)
#define RGBA_YELLOWDK(alpha) SDL_MapRGBA(RGB_getFormat(), 128, 128, 0, alpha)
#define RGBA_YELLOW(alpha) SDL_MapRGBA(RGB_getFormat(), 255, 255, 0, alpha)
#define RGBA_YELLOWLT(alpha) SDL_MapRGBA(RGB_getFormat(), 255, 255, 128, alpha)
#define RGBA_YELLOWVLT(alpha) SDL_MapRGBA(RGB_getFormat(), 255, 255, 192, alpha)


#define RGBA_ORANGEVDK(alpha) SDL_MapRGBA(RGB_getFormat(), 180, 64, 0, alpha)
#define RGBA_ORANGEDK(alpha) SDL_MapRGBA(RGB_getFormat(), 224, 96, 0, alpha)
#define RGBA_ORANGE(alpha) SDL_MapRGBA(RGB_getFormat(), 255, 128, 0, alpha)
#define RGBA_ORANGELT(alpha) SDL_MapRGBA(RGB_getFormat(), 255, 160, 32, alpha)
#define RGBA_ORANGEVLT(alpha) SDL_MapRGBA(RGB_getFormat(), 255, 192, 64, alpha)


#define RGBA_BROWNVDK(alpha) SDL_MapRGBA(RGB_getFormat(), 32, 16, 0, alpha)
#define RGBA_BROWNDK(alpha) SDL_MapRGBA(RGB_getFormat(), 64, 32, 0, alpha)
#define RGBA_BROWN(alpha) SDL_MapRGBA(RGB_getFormat(), 128, 64, 0, alpha)
#define RGBA_BROWNLT(alpha) SDL_MapRGBA(RGB_getFormat(), 192, 128, 32, alpha)
#define RGBA_BROWNVLT(alpha) SDL_MapRGBA(RGB_getFormat(), 255, 192, 64, alpha)

#define RGBA_TEALVDK(alpha) SDL_MapRGBA(RGB_getFormat(), 0, 64, 64, alpha)
#define RGBA_TEALDK(alpha) SDL_MapRGBA(RGB_getFormat(), 0, 128, 128, alpha)
#define RGBA_TEAL(alpha) SDL_MapRGBA(RGB_getFormat(), 0, 255, 255, alpha)
#define RGBA_TEALLT(alpha) SDL_MapRGBA(RGB_getFormat(), 128, 255, 255, alpha)
#define RGBA_TEALVLT(alpha) SDL_MapRGBA(RGB_getFormat(), 192, 255, 255, alpha)

#define RGBA_TURQUOISEVDK(alpha) SDL_MapRGBA(RGB_getFormat(), 0, 64, 64, alpha)
#define RGBA_TURQUOISEDK(alpha) SDL_MapRGBA(RGB_getFormat(), 0, 128, 128, alpha)
#define RGBA_TURQUOISE(alpha) SDL_MapRGBA(RGB_getFormat(), 0, 255, 255, alpha)
#define RGBA_TURQUOISELT(alpha) SDL_MapRGBA(RGB_getFormat(), 128, 255, 255, alpha)
#define RGBA_TURQUOISEVLT(alpha) SDL_MapRGBA(RGB_getFormat(), 192, 255, 255, alpha)

#define RGBA_BLACK(alpha) SDL_MapRGBA(RGB_getFormat(), 0, 0, 0, alpha)
#define RGBA_BLACK1(alpha) SDL_MapRGBA(RGB_getFormat(), 1, 1, 1, alpha)
#define RGBA_BLACK2(alpha) SDL_MapRGBA(RGB_getFormat(), 2, 2, 2, alpha)

#define RGBA_GRAYVDK(alpha) SDL_MapRGBA(RGB_getFormat(), 32, 32, 32, alpha)
#define RGBA_GRAYDK(alpha) SDL_MapRGBA(RGB_getFormat(), 64, 64, 64, alpha)
#define RGBA_GRAY(alpha) SDL_MapRGBA(RGB_getFormat(), 128, 128, 128, alpha)
#define RGBA_GRAYLT(alpha) SDL_MapRGBA(RGB_getFormat(), 192, 192, 192, alpha)
#define RGBA_GRAYVLT(alpha) SDL_MapRGBA(RGB_getFormat(), 224, 224, 224, alpha)

#define RGBA_WHITE(alpha) SDL_MapRGBA(RGB_getFormat(), 255, 255, 255, alpha)
#define RGBA_WHITE1(alpha) SDL_MapRGBA(RGB_getFormat(), 254, 254, 254, alpha)
#define RGBA_WHITE2(alpha) SDL_MapRGBA(RGB_getFormat(), 253, 253, 253, alpha)
#define RGBA_WHITEOFF(alpha) SDL_MapRGBA(RGB_getFormat(), 255, 255, 224, alpha)



#ifdef __cplusplus
} /*extern "C"*/
#endif


#endif
